<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once 'conexion.php';

function getUnidadId($conn, $abrev) {
    $stmt = $conn->prepare("SELECT id FROM unidadmedida WHERE abrev = :abrev LIMIT 1");
    $stmt->execute([':abrev' => trim($abrev)]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? $row['id'] : null;
}

function getUnidadAbrevUniv($conn, $abrev) {
    $stmt = $conn->prepare("SELECT abrev_universal  FROM unidadmedida WHERE abrev = :abrev LIMIT 1");
    $stmt->execute([':abrev' => trim($abrev)]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? $row['abrev_universal'] : null;
}

function getMarcaId($conn, $nombre) {
    $stmt = $conn->prepare("SELECT id FROM marcas WHERE nombre ILIKE :nombre LIMIT 1");
    $stmt->execute([':nombre' => trim($nombre)]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? $row['id'] : null;
}

function getModeloId($conn, $nombre) {
    $stmt = $conn->prepare("SELECT id FROM modelos WHERE nombre ILIKE :nombre LIMIT 1");
    $stmt->execute([':nombre' => trim($nombre)]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ? $row['id'] : null;
}

function importarProductos($archivoCSV, $conn, $empresaId = 1) {
    echo "<h4>🟦 Procesando productos...</h4>";
    $file = fopen($archivoCSV, 'r');
    fgetcsv($file, 0, ';');
    $insertados = 0;

    $sql = "INSERT INTO productos (
        codigo_producto, nombre, descripcion, afecto_igv, porcentaje_igv,
        pcompra_ant, ult_pcompra, peso, stock_minimo, pprecio_menor, pprecio_mayor, pprecio_dist,
        precmenor_und, precmayor_und, precdist_und, precmenor_fra, precmayor_fra, precdist_fra,
        unidad_id, fraccion_id, capacidad, categoria_id, marca_id, modelo_id, activo, usa_serie,
        offsystem, empresa_id
    ) VALUES (
        :codigo_producto, :nombre, :descripcion, :afecto_igv, :porcentaje_igv,
        :pcompra_ant, :ult_pcompra, :peso, :stock_minimo, :pprecio_menor, :pprecio_mayor, :pprecio_dist,
        :precmenor_und, :precmayor_und, :precdist_und, :precmenor_fra, :precmayor_fra, :precdist_fra,
        :unidad_id, :fraccion_id, :capacidad, :categoria_id, :marca_id, :modelo_id, :activo, :usa_serie,
        :offsystem, :empresa_id
    ) RETURNING id";

    $stmt = $conn->prepare($sql);

    while (($line = fgetcsv($file, 0, ';')) !== false) {
        $line = array_map(function($v) {
            return mb_convert_encoding(str_replace('"', '', $v), 'UTF-8', 'auto');
        }, $line);

        $codigo = str_pad(trim($line[0]), 10, "0", STR_PAD_LEFT);
        $nombre = trim($line[2]);
        $inactivo = isset($line[46]) ? trim($line[46]) : '1';

        if ($codigo === '' || $nombre === '') {
            echo "<p style='color:gray'>⛔ Producto omitido por falta de código o nombre.</p>";
            continue;
        }

        if ($inactivo !== '0') {
            echo "<p style='color:orange'>🚫 Producto inactivo: $codigo - $nombre</p>";
            continue;
        }

        $unidad_id = $line[3];
        $fraccion_id = $line[4];
        $capacidad = $line[5];
        $marca_id = $line[17];
        $modelo_id = $line[18];

        if (!$unidad_id || !$fraccion_id || !$marca_id || !$modelo_id) {
            echo "<p style='color:red'>⚠️ $codigo - $nombre descartado: unidad=$unidad_id, fracción=$fraccion_id, marca=$marca_id, modelo=$modelo_id</p>";
            continue;
        }

        $check = $conn->prepare("SELECT 1 FROM productos WHERE descripcion = :descripcion LIMIT 1");
        $check->execute([':descripcion' => $nombre]);
        if ($check->fetch()) {
            echo "<p style='color:orange'>⚠️ Ya existe: $nombre — se omite</p>";
            continue;
        }

        $stmt->execute([
            ':codigo_producto' => $codigo,
            ':nombre' => $nombre,
            ':descripcion' => $nombre,
            ':afecto_igv' => 0,
            ':porcentaje_igv' => 0.00,
            ':pcompra_ant' => floatval($line[8]),
            ':ult_pcompra' => floatval($line[11]),
            ':peso' => floatval($line[14]),
            ':stock_minimo' => floatval($line[38]),
            'pprecio_menor' => floatval($line[22]), 
            'pprecio_mayor' => floatval($line[23]), 
            'pprecio_dist' => floatval($line[24]),
            ':precmenor_und' => floatval($line[26]),
            ':precmayor_und' => floatval($line[27]),
            'precdist_und' => floatval($line[28]), 
            'precmenor_fra' => floatval($line[30]), 
            'precmayor_fra' => floatval($line[31]), 
            'precdist_fra' => floatval($line[32]),
            ':unidad_id' => $unidad_id,
            ':fraccion_id' => $fraccion_id,
            ':capacidad' => $capacidad,
            ':categoria_id' => 1,
            ':marca_id' => 1,
            ':modelo_id' => 1,
            ':activo' => 1,
            ':usa_serie' => 0,
            ':offsystem' => 0,
            ':empresa_id' => $empresaId
        ]);

        $insertados++;
    }

    fclose($file);
    echo "<p style='color:green'>✅ Productos insertados: $insertados</p>";
}

function importarEquivalencias($archivoCSV, $conn, $empresaId = 1) {
    echo "<h4>🟦 Procesando equivalencias...</h4>";
    $file = fopen($archivoCSV, 'r');
    fgetcsv($file, 0, ';');
    $insertados = 0;

    $sql = "INSERT INTO equivalencias (
        producto_id, nombre, capacidad_precio, abrev_precio, abrev_universal,
        precio_venta, empresa_id, sucursal_id, codigo_producto
    ) VALUES (
        :producto_id, :nombre, :capacidad_precio, :abrev_precio, :abrev_universal,
        :precio_venta, :empresa_id, :sucursal_id, :codigo_producto
    )";

    $stmt = $conn->prepare($sql);

    while (($line = fgetcsv($file, 0, ';')) !== false) {
        $codigo = trim($line[1]); // IdProducto
        $nombre = trim($line[3]); // NomPrecio
        $capacidad = floatval($line[4]); // CapaPrec
        $abrev_precio = trim($line[5]); // AbrevPrec
        $precio = floatval($line[6]); // Precio_Publico

        // Buscar producto ya insertado
        $productoStmt = $conn->prepare("SELECT id FROM productos WHERE codigo_producto = :codigo LIMIT 1");
        $productoStmt->execute([':codigo' => $codigo]);
        $producto = $productoStmt->fetch(PDO::FETCH_ASSOC);
        if (!$producto) {
            echo "<p style='color:red'>❌ Producto no encontrado para equivalencia: $codigo</p>";
            continue;
        }

        $stmt->execute([
            ':producto_id' => $producto['id'],
            ':nombre' => $nombre,
            ':capacidad_precio' => $capacidad,
            ':abrev_precio' => $abrev_precio,
            ':abrev_universal' => getUnidadAbrevUniv($conn, $abrev_precio),
            ':precio_venta' => $precio,
            ':empresa_id' => $empresaId,
            ':sucursal_id' => 1,
            ':codigo_producto' => $codigo
        ]);

        $insertados++;
    }

    fclose($file);
    echo "<p style='color:green'>✅ Equivalencias insertadas: $insertados</p>";
}

function importarKardex($archivoCSV, $conn, $empresaId = 1) {
    echo "<h4>🟦 Procesando stock inicial (kardex)...</h4>";
    $file = fopen($archivoCSV, 'r');
    fgetcsv($file, 0, ';');
    $insertados = 0;

    $sql = "INSERT INTO kardex (
        nserie, ndocumento, fecha_emision, fecha_traslado, operacion,
        producto_id, cantidad, costo, unidad_medida, docfiscal_id,
        empresa_id, sucursal_id, almacen_id
    ) VALUES (
        :nserie, :ndocumento, :fecha_emision, :fecha_traslado, :operacion,
        :producto_id, :cantidad, :costo, :unidad_medida, :docfiscal_id,
        :empresa_id, :sucursal_id, :almacen_id
    )";

    $stmt = $conn->prepare($sql);

    while (($line = fgetcsv($file, 0, ';')) !== false) {
        $stockActual = floatval($line[6]);
        if ($stockActual <= 0) {
            echo "<p style='color:gray'>⛔ Stock 0 → omitido.</p>";
            continue;
        }

        $codigo = str_pad(trim($line[2]), 10, "0", STR_PAD_LEFT);
        $almacen = trim($line[1]) === 'PR' ? 1 : 2;

        $productoStmt = $conn->prepare("SELECT id FROM productos WHERE codigo_producto = :codigo LIMIT 1");
        $productoStmt->execute([':codigo' => $codigo]);
        $producto = $productoStmt->fetch(PDO::FETCH_ASSOC);

        if (!$producto) {
            echo "<p style='color:red'>❌ Producto no encontrado para stock: $codigo</p>";
            continue;
        }

        $stmt->execute([
            ':nserie' => 'INV1',
            ':ndocumento' => $almacen == 1 ? '00000001' : '00000002',     //.str_pad($insertados+1, 6, '0', STR_PAD_LEFT),
            ':fecha_emision' => date('Y-m-d'),
            ':fecha_traslado' => date('Y-m-d'),
            ':operacion' => 'INV',
            ':producto_id' => $producto['id'],
            ':cantidad' => $stockActual,
            ':costo' => floatval($line[5]),
            ':unidad_medida' => $line[10],
            ':docfiscal_id' => 11,
            ':empresa_id' => $empresaId,
            ':sucursal_id' => 1,
            ':almacen_id' => $almacen
        ]);

        $insertados++;
    }

    fclose($file);
    echo "<p style='color:green'>✅ Registros en kardex: $insertados</p>";
}

function insertarCostosDesdeProductos($conn) {
    echo "<h4>🟦 Insertando costos iniciales...</h4>";

    $fecha = date('Y-m-d');
    $compra_id = 0;
    $almacenes = [1]; // puedes ajustar si usas solo uno

    $stmt = $conn->prepare("
        INSERT INTO costos (
            almacen_id, producto_id, costo, costo_promedio, flete, estiba, fecha, compra_id
        ) VALUES (
            :almacen_id, :producto_id, :costo, :costo_promedio, 0.00, 0.00, :fecha, :compra_id
        ) ON CONFLICT DO NOTHING
    ");

    $productos = $conn->query("SELECT id, ult_pcompra FROM productos WHERE activo = true");

    $insertados = 0;
    foreach ($productos as $prod) {
        foreach ($almacenes as $almacen_id) {
            $stmt->execute([
                ':almacen_id' => $almacen_id,
                ':producto_id' => $prod['id'],
                ':costo' => $prod['ult_pcompra'],
                ':costo_promedio' => $prod['ult_pcompra'],
                ':fecha' => $fecha,
                ':compra_id' => $compra_id
            ]);
            $insertados++;
        }
    }

    echo "<p style='color:green'>✅ Costos registrados: $insertados</p>";
}

function insertarEquivalenciasFaltantes($conn, $empresaId = 1) {
    echo "<h4>🟦 Insertando equivalencias faltantes...</h4>";

    $sql = "
        SELECT p.id AS producto_id, p.codigo_producto, p.precmenor_und, p.capacidad,
               p.unidad_id, p.fraccion_id
        FROM productos p
        WHERE p.activo = true
          AND NOT EXISTS (
              SELECT 1 FROM equivalencias e
              WHERE e.producto_id = p.id
          )
    ";

    $productos = $conn->query($sql)->fetchAll(PDO::FETCH_ASSOC);
    $insertados = 0;

    $stmt = $conn->prepare("
        INSERT INTO equivalencias (
            producto_id, nombre, capacidad_precio, abrev_precio, abrev_universal,
            precio_venta, empresa_id, sucursal_id, codigo_producto
        ) VALUES (
            :producto_id, :nombre, :capacidad_precio, :abrev_precio, :abrev_universal,
            :precio_venta, :empresa_id, :sucursal_id, :codigo_producto
        )
    ");

    foreach ($productos as $prod) {
        $producto_id = $prod['producto_id'];
        $codigo = $prod['codigo_producto'];
        $capacidad = floatval($prod['capacidad']);
        $unidad_id = $prod['unidad_id'];
        $fraccion_id = $prod['fraccion_id'];
        $precio = floatval($prod['precmenor_und']);

        // Obtener unidad base
        $unidadStmt = $conn->prepare("SELECT nombre, abrev, abrev_universal FROM unidadmedida WHERE id = :id");
        $unidadStmt->execute([':id' => $unidad_id]);
        $unidadData = $unidadStmt->fetch(PDO::FETCH_ASSOC);

        if (!$unidadData) {
            echo "<p style='color:red'>❌ Unidad no encontrada para producto $codigo</p>";
            continue;
        }

        $unidadNombre = $unidadData['nombre'];
        $abrev = substr($unidadData['abrev'], 0, 4);
        $abrevUniv = substr($unidadData['abrev_universal'], 0, 4);

        // Primera equivalencia
        $stmt->execute([
            ':producto_id' => $producto_id,
            ':nombre' => $unidadNombre,
            ':capacidad_precio' => 1,
            ':abrev_precio' => $abrev,
            ':abrev_universal' => $abrevUniv,
            ':precio_venta' => $precio,
            ':empresa_id' => $empresaId,
            ':sucursal_id' => 1,
            ':codigo_producto' => $codigo
        ]);
        $insertados++;

        // Segunda equivalencia SOLO si capacidad > 1
        if ($capacidad > 1) {
            $fraccionStmt = $conn->prepare("SELECT abrev FROM unidadmedida WHERE id = :id");
            $fraccionStmt->execute([':id' => $fraccion_id]);
            $fraccion = $fraccionStmt->fetchColumn();
            $fraccion = substr($fraccion ?: '', 0, 4);

            $nombre_compuesto = "$unidadNombre x $capacidad $fraccion";

            $stmt->execute([
                ':producto_id' => $producto_id,
                ':nombre' => $nombre_compuesto,
                ':capacidad_precio' => $capacidad,
                ':abrev_precio' => $abrev,
                ':abrev_universal' => $abrevUniv,
                ':precio_venta' => $precio,
                ':empresa_id' => $empresaId,
                ':sucursal_id' => 1,
                ':codigo_producto' => $codigo
            ]);
            $insertados++;
        }
    }

    echo "<p style='color:green'>✅ Equivalencias generadas: $insertados</p>";
}

function limpiarEquivalenciasDuplicadasCapacidad1($conn) {
    echo "<h4>🧹 Corrigiendo equivalencias duplicadas con capacidad = 1...</h4>";

    $sql = "
        DELETE FROM equivalencias e
        USING (
            SELECT producto_id
            FROM equivalencias
            GROUP BY producto_id
            HAVING COUNT(*) > 1
            INTERSECT
            SELECT id
            FROM productos
            WHERE capacidad = 1
        ) AS p
        WHERE e.producto_id = p.producto_id
        AND e.id NOT IN (
            SELECT MIN(id)
            FROM equivalencias
            WHERE producto_id = p.producto_id
            GROUP BY producto_id
        );
    ";

    try {
        $conn->exec($sql);
        echo "<p style='color:green'>✅ Equivalencias incorrectas eliminadas.</p>";
    } catch (PDOException $e) {
        echo "<p style='color:red'>❌ Error al limpiar equivalencias: " . $e->getMessage() . "</p>";
    }
}

// EJECUTAR
try {
    importarProductos(__DIR__ . "/Productos_FerBarboza.csv", $conn);
    importarEquivalencias(__DIR__ . "/Equivalencias.csv", $conn);
    importarKardex(__DIR__ . "/Stock_FerBarboza.csv", $conn);
    insertarCostosDesdeProductos($conn);
    insertarEquivalenciasFaltantes($conn);
    //limpiarEquivalenciasDuplicadasCapacidad1($conn);
    echo "<h4 style='color:blue'>🏁 Proceso de importación finalizado.</h4>";
} catch (Exception $e) {
    echo "<p style='color:red'>❌ Error: ".$e->getMessage()."</p>";
}