<?php
error_reporting(E_ALL ^ E_NOTICE);

header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Content-Type: application/json; charset=utf-8");

// Incluir la conexión a PostgreSQL
include_once '../conexion.php';
include_once '../clases/TokenValidator.php';

// Llama la validación antes de cualquier consulta
$usuarioId = TokenValidator::validar($conn);

$bodyRequest = file_get_contents("php://input");
$datos = json_decode($bodyRequest, true);

// Validar que la fecha existe y tiene formato correcto (YYYY-MM-DD)
if (!isset($datos[0]) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $datos[0])) {
    error_response("Fecha inválida.");
}

$fecha = $datos[0];

try {
    // Consulta con parámetro seguro
    $sql = "SELECT * FROM tipo_cambio WHERE fecha = :fecha AND activo = TRUE LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->bindParam(':fecha', $fecha, PDO::PARAM_STR);
    $stmt->execute();
    
    // Obtener el resultado
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    // Enviar la respuesta en JSON
    print_json($row ? [$row] : []);

} catch (PDOException $e) {
    error_response("Error en la consulta: " . $e->getMessage());
}

// Cerrar la conexión
$conn = null;

function print_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE);

    switch (json_last_error()) {
        case JSON_ERROR_NONE:
            break;
        case JSON_ERROR_DEPTH:
            error_response('Maximum stack depth exceeded');
            break;
        case JSON_ERROR_STATE_MISMATCH:
            error_response('Underflow or the modes mismatch');
            break;
        case JSON_ERROR_CTRL_CHAR:
            error_response('Unexpected control character found');
            break;
        case JSON_ERROR_SYNTAX:
            error_response('Syntax error, malformed JSON');
            break;
        case JSON_ERROR_UTF8:
            error_response('Malformed UTF-8 characters, possibly incorrectly encoded');
            break;
        default:
            error_response('Unknown error');
            break;
    }
}

function error_response($message) {
    http_response_code(500);
    echo json_encode(['error' => $message], JSON_UNESCAPED_UNICODE);
    exit();
}
?>
