<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once __DIR__ . '/../../conexion.php';
require_once __DIR__ . '/../../tcpdf/tcpdf.php';

header("Content-Type: application/pdf; charset=utf-8");

try {
    $bodyRequest = file_get_contents("php://input");
    $datos = json_decode($bodyRequest, true);

    if (!isset($datos[0], $datos[1], $datos[2], $datos[3])) {
        throw new Exception("Parámetros inválidos. Se espera [fecha_inicio, fecha_fin].");
    }

    $fechaInicio = $datos[0];
    $fechaFin    = $datos[1];
    $empresaId   = (int) $datos[2];
    $sucursalId  = (int) $datos[3];

    $sql = "SELECT 
                CASE tipo_pago
                    WHEN 1 THEN 'PAGO TARJETA'
                    WHEN 2 THEN 'PAGO YAPE'
                    WHEN 3 THEN 'PAGO PLIN'
                    WHEN 4 THEN 'PAGO IZIPAY'
                    WHEN 5 THEN 'PAGO OTROS'
                END AS tipo,
                SUM(monto_pago) AS total_pago
            FROM despacho_pago
            WHERE tipo_pago IN (1, 2, 3, 4, 5)
              AND fecha_registro BETWEEN :fecha_inicio AND :fecha_fin::timestamp + time '23:59:00'
              AND empresa_id = :empresa_id
              AND sucursal_id = :sucursal_id
            GROUP BY tipo_pago
            ORDER BY tipo_pago";

    $stmt = $conn->prepare($sql);
    $stmt->execute([
        ':fecha_inicio' => $fechaInicio,
        ':fecha_fin'    => $fechaFin,
        ':empresa_id'   => $empresaId,
        ':sucursal_id'  => $sucursalId
    ]);
    $pagos = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($pagos)) {
        throw new Exception("No se encontraron pagos electrónicos registrados en el rango de fechas indicado.");
    }

    $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
    $pdf->SetMargins(10, 10, 10);
    $pdf->SetAutoPageBreak(true, 10);
    $pdf->SetDisplayMode(100, 'SinglePage');
    $pdf->AddPage();

    $logo_path = '../../tcpdf/logos/10278514876.png';
    if (file_exists($logo_path)) {
        $pdf->Image($logo_path, '', '', 30, 15, '', '', 'T', false, 300, 'C', false, false, 0, false, false, false);
    }
    $pdf->Ln(18);

    // Cabecera
    $pdf->SetFont('helvetica', 'B', 14);
    $pdf->Cell(0, 10, 'REPORTE DE PAGOS ELECTRÓNICOS', 0, 1, 'C');
    $pdf->SetFont('helvetica', '', 10);
    $pdf->Cell(0, 6, "Del $fechaInicio al $fechaFin", 0, 1, 'C');
    $pdf->Ln(5);

    // Tabla
    $pdf->SetFillColor(5, 152, 172);       // Fondo celeste oscuro
    $pdf->SetTextColor(255, 255, 255);     // Letras blancas
    $pdf->SetDrawColor(0, 0, 0);           // Borde negro
    $pdf->SetFont('helvetica', 'B', 10);

    $pdf->Cell(120, 8, 'Tipo de Pago', 1, 0, 'C', true);  // true = aplicar color de fondo
    $pdf->Cell(70, 8, 'Total Pagado (S/)', 1, 1, 'C', true);

    // Luego, restablece el color del texto a negro para el cuerpo
    $pdf->SetTextColor(0, 0, 0);

    $pdf->SetFont('helvetica', '', 10);

    $totalGeneral = 0;

    foreach ($pagos as $pago) {
        $pdf->Cell(120, 7, utf8_decode($pago['tipo']), 1, 0, 'L');
        $pdf->Cell(70, 7, number_format($pago['total_pago'], 2), 1, 1, 'R');

        $totalGeneral += (float) $pago['total_pago'];;
    }
    // Línea vacía antes del total
    $pdf->Ln(2);

    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(120, 8, 'TOTAL GENERAL', 1, 0, 'R');
    $pdf->Cell(70, 8, number_format($totalGeneral, 2), 1, 1, 'R');

    $pdf->Output('reporte_pagos_electronicos.pdf', 'I');

} catch (Exception $e) {
    error_log("ERROR: " . $e->getMessage());
    echo json_encode([
        "estado" => "0",
        "mensaje" => $e->getMessage()
    ]);
}
?>
